
require "InterfaceScriptCommonLibrary"
json = require "json"

------------------------------------------------------------------------------
-- Hilfsfunktionen
------------------------------------------------------------------------------

function cvtStringToNumber( strNumber )
    local strValue = string.gsub(strNumber, ",", "%.")
    return tonumber(strValue)
end

function split(inputstr, sep)
    if sep == nil then
            sep = "%s"
    end
    local t={} ; i=1
    for str in string.gmatch(inputstr, "([^"..sep.."]+)") do
            t[i] = str
            i = i + 1
    end
    return t
end

function formatHTTPText( strHTTPText )
    return 
        string.gsub( strHTTPText, "(\\+)(%d+)", 

            function (n, m) 
                if (0 == math.fmod(string.len(n),2)) then
                    -- Anzahl der \ ist gerade
                    return string.gsub(n,"\\\\","\\") .. m 
                else
                    -- Anzahl der \ ist ungerade, aber mindestens 1
                    n = string.sub(n,1,-2)
                    return string.gsub(n,"\\\\","\\") .. string.char(math.fmod(tonumber(m),256) )
                end
            end)
end

function trimBOMIfNeeded( strRaw )
	if(string.sub(strRaw, 1, 3) == "﻿") then
		return string.sub(strRaw, 4, -1)
	end
	return strRaw
end

-- Dump Tabelle: Erzeugt einen String aus einer Tabelle und gibt diesen zurück
function dump(o)
    if type(o) == 'table' then
       local s = '{ '
       for k,v in pairs(o) do
          if type(k) ~= 'number' then k = '"'..k..'"' end
          s = s .. '['..k..'] = ' .. dump(v) .. ','
       end
       return s .. '} '
    else
       return tostring(o)
    end
 end

-- Gibt die Länge der Tabelle zurück.
function getTableLength(T)
    local count = 0
    for _ in pairs(T) do count = count + 1 end
    return count
end


-- ***************************************************************************
-- Prozessankopplung
-- ***************************************************************************

-- ===========================================================================
-- Initialisierung, einmaliger Aufruf vor dem Start der Pollfunktion
-- ===========================================================================

local lastPollTimestamp
local intervallInSeconds
local lastPollTimestampAstro
local dateTimeSunrise
local dateTimeSunset 


function Init()
    E:trace("---- HTTP Befehle: Init Start ----")

    -- Initialisieren der Ressourcen
    lastPollTimestamp = os.clock()
    lastPollTimestampAstro = os.clock()
    HTTP_RQ = E.ResourceTable["HTTP"]
    HTTP_RQ:Open()
	
	GetValuesOfWeatherAPI()

    E:trace("---- HTTP Befehle: Init Ende ----")
end

-- ===========================================================================
-- Ruft die Weather API per HTTP auf und wertet das Ergebnis aus.
-- ===========================================================================

--Sonderbehandlung für die alte API
function EncodeWeatherURL( strURL )

	local a,b
	a,b = string.find(strURL, "q=.*&") 
	-- E:trace( "a: " ..tostring(a) .." b: " ..tostring(b) )
	if ((nil == a) or (nil == b)) then 
		return strURL;
	end
	
	local strLocation = string.sub(strURL,a+2,b-1)
	-- E:trace( "Location: " ..tostring(strLocation) )s
	local strLocationUTF = HTTP_RQ:ConvertASCIIToUTF8(strLocation)
	-- E:trace( "Location UTF: " ..tostring(strLocationUTF) )
	local strEncLocation = HTTP_RQ:ConvertToURLEncoding(strLocationUTF)
	-- E:trace( "Encoded location: " ..tostring(strEncLocation) )
	
	strEncLocation = string.gsub( strEncLocation, "%%", "%%%%" )
	return string.gsub(strURL, "q=.*&", "q="..tostring(strEncLocation).."&" ) 
end

function GetValuesOfWeatherAPI()
    E:trace("Rufe Werte von der API ab...")
    -- Url
    local oHTTPDevice = E.PVTable["httpdevices"]

    local strPollIntervall = oHTTPDevice["pollSeconds"]:GetValue()

    -- Überprüfe PollIntervall nach NIL
    if (("" == strPollIntervall) or (nil == strPollIntervall) or (cvtStringToNumber(strPollIntervall) < 60)) then
        intervallInSeconds = 3600
        oHTTPDevice["pollSeconds"]:SetValue(intervallInSeconds, constValueChange)
    else
        intervallInSeconds = cvtStringToNumber(strPollIntervall)
    end


    local currentDate = GetValuesForCurrentWeather(oHTTPDevice)
    GetValuesForForecast(oHTTPDevice, currentDate)
    E:trace("Werte aktualisiert")
end


function SendHttpRequest(strWeatherUri)
    if (("" == strWeatherUri) or (nil == strWeatherUri)) then
        E:trace( "GetValuesOfWeatherAPI, unbekannte Url" )
        return nil
    end
	--strWeatherUri = EncodeWeatherURL( strWeatherUri )
	-- E:trace( "strWeatherUri: " ..tostring(strWeatherUri) )
	
	-- Sende HTTP GET Request, erwarte Json Ausdruck als Antwort
	HTTP_RQ:SetURL( strWeatherUri )
	HTTP_RQ:SendRequest( 0, "" )
	
	local nReturnCode, strError, nStatusCode, strData
	nReturnCode, strError, nStatusCode, strData = HTTP_RQ:GetRxData()	

	--E:trace( "cURL Return: " ..tostring(nReturnCode) ..", Error: " ..strError ..", HTTP Status: " ..tostring(nStatusCode) )
    --E:trace( "Response: " ..strData  )
    E:trace( "Error: " ..strError ..", HTTP Status: " ..tostring(nStatusCode) )
    
    -- Überprüfe Binary Order Mark und falls vorhanden, entferne es
	strData = HTTP_RQ:ConvertUTF8ToASCII( trimBOMIfNeeded(strData) )
	
    -- Decodiere Json zu einer Tabelle
    local err, decodedJsonTable = pcall(DecodeJson,strData)
    if err == false then
        E:trace("Keine Antwort vom Server oder ungueltige URL")
        return nil
    end
    return decodedJsonTable
end

function GetValuesForCurrentWeather(oHTTPDevice)
    local strWeatherUri  = oHTTPDevice["weatherUriCurrent"]:GetValue()
    local decodedJsonTable = SendHttpRequest(strWeatherUri)
    
    if (decodedJsonTable == nil) then
        return
    end
	-- Set Location Werte
	SetLocationTable(decodedJsonTable)
	
	-- Set Aktuelle Werte
	local currentDate = SetCurrentTable(decodedJsonTable)
	
	-- Set Astro Werte
    SetAstroTable(decodedJsonTable)
    
    return currentDate
end

function GetValuesForForecast(oHTTPDevice, currentDate)
    local strWeatherUri  = oHTTPDevice["weatherUri"]:GetValue()
    local decodedJsonTable = SendHttpRequest(strWeatherUri)
    
    if (decodedJsonTable == nil) then
        return
    end  
	-- Set Location Werte
	--SetLocationTable(decodedJsonTable)
	   
    if (decodedJsonTable["list"] ~= nil) then
        -- Set Vorhersage
        SetForecastDay(decodedJsonTable, currentDate)
    end
end

function DecodeJson(strValue)
    return json.decode(strValue)
end

function SetLocationTable(decodedJsonTable)
    local locationTable = E.PVTable["locationRoot"]
	local timezoneh = (decodedJsonTable["timezone"])/3600
	
	locationTable["lName"]:SetValue(tostring(decodedJsonTable["name"]), constValueChange)
	locationTable["lCountry"]:SetValue(tostring(decodedJsonTable["sys"]["country"]), constValueChange)
	locationTable["lLatitude"]:SetValue(decodedJsonTable["coord"]["lat"], constValueChange)
	locationTable["lLongitude"]:SetValue(decodedJsonTable["coord"]["lon"], constValueChange)
	locationTable["lTz_id"]:SetValue(timezoneh, constValueChange)
end

function SetForecastDay(decodedJsonTable, currentDateTime)
	
	-- Vorhersage
    local forecastRootTable = E.PVTable["forecastRoot"] 

    -- Aktuelles Wetter
    if currentDateTime == nil or currentDateTime == "" then
        E:trace("Vorhersagewerte konnten nicht gesetzt werden: Aktuelles Datum wurde nicht richtig gelesen.")
        return
    end
    local currentDate = split(currentDateTime, " ")[1]
    

	-- Tagesvorhersage (Enthält mehrere Tage)
    local forecastRootDays = forecastRootTable["forecastDayRoot"] 

    -- Vorhersage des Tages (mehrere Tage verfügbar!), 
	-- Lade die Tabelle der Tage
    local forecastDays = forecastRootDays["forecastDay"] 
	-- Anzahl der Tage die der Benutzer angelegt hat
    local userNumberOfDays = getTableLength(forecastDays) 
    -- Anzahl der Tage die die Wetter API gesendet hat
    
    -- Sortiere die Daten nach Tagen
    local sortedDays = SortListByDays(decodedJsonTable)
    
    -- Entferne den aktuellen Tag
    if getTableLength(sortedDays) > 1 then
        if sortedDays[1].date == currentDate then
            table.remove(sortedDays, 1)
        end
    end

    local apiNumberOfDays = getTableLength(sortedDays) 

    -- Falls die API weniger Tage gesendet hat als der Benutzer angelegt hat, 
	-- nimm die Anzahl der API Tage. (Verhindert IndexOutOfRange Exception)
    if (apiNumberOfDays < userNumberOfDays) then
        userNumberOfDays = apiNumberOfDays;
    end
  
    -- Für jeden Tag die Werte eintragen
    for i=1, userNumberOfDays, 1 do
        if i < 5 then
            local forecastDay = forecastDays[i]
            local dayValues = sortedDays[i].values
            
            local minTemp = nil
            local maxTemp = nil
            local sumRain = 0
            local sumSnow = 0
            local dayPressure = 0
            local dayHumidity = 0
            local dayWind = 0
            local dayWind = ""

            forecastDay["fdate"]:SetValue(sortedDays[i].date, constValueChange)

            --Setze die Werte für die Detailansicht
            for detailDay=1, getTableLength(dayValues), 1 do
                local detailTable = forecastDay["detail" .. detailDay]
                if detailTable == nil then
                    E:trace("Details " .. detailDay .. "Ordner nicht vorhanden")
                else
                    detailTable["fdate" .. detailDay]:SetValue(dayValues[detailDay].dateTime, constValueChange)
                    detailTable["dMaxTemp_c" .. detailDay]:SetValue(dayValues[detailDay].dMaxTemp_c, constValueChange)
                    detailTable["dMinTemp_c".. detailDay]:SetValue(dayValues[detailDay].dMinTemp_c, constValueChange)
                    detailTable["dHumidity".. detailDay]:SetValue(dayValues[detailDay].dHumidity, constValueChange)
                    detailTable["dPressure".. detailDay]:SetValue(dayValues[detailDay].dPressure, constValueChange)		
                    detailTable["dWind_mps".. detailDay]:SetValue(dayValues[detailDay].dWind_mps, constValueChange)
                    detailTable["dWind_deg" .. detailDay]:SetValue(dayValues[detailDay].dWind_deg, constValueChange)
                    detailTable["dIcon" .. detailDay]:SetValue(dayValues[detailDay].dIcon, constValueChange)     
                    detailTable["dCloud_desc" .. detailDay]:SetValue(dayValues[detailDay].dCloud_desc, constValueChange)  	
                    detailTable["dCloud" .. detailDay]:SetValue(dayValues[detailDay].dCloud, constValueChange)		
                    detailTable["dRain" .. detailDay]:SetValue(dayValues[detailDay].dRain, constValueChange)
                    detailTable["dSnow" .. detailDay]:SetValue(dayValues[detailDay].dSnow, constValueChange)

                    local nRain =  cvtStringToNumber(dayValues[detailDay].dRain)
                    local nSnow = cvtStringToNumber(dayValues[detailDay].dSnow)

                    if nRain ~= nil then
                        sumRain = sumRain + nRain
                    end

                    if nSnow ~= nil then
                        sumSnow = sumSnow + nSnow
                    end

                    local detailMinTemp = cvtStringToNumber(dayValues[detailDay].dMinTemp_c)
                    local detailMaxTemp = cvtStringToNumber(dayValues[detailDay].dMaxTemp_c)

                    if minTemp == nil or minTemp > detailMinTemp then
                        minTemp = detailMinTemp
                    end
                    if maxTemp == nil or maxTemp < detailMaxTemp then
                        maxTemp = detailMaxTemp
                    end

                    if detailDay == 5 then
                        dayHumidity = dayValues[detailDay].dHumidity
                        dayPressure = dayValues[detailDay].dPressure
                        dayWind = dayValues[detailDay].dWind_mps
                        dayIcon = dayValues[detailDay].dIcon
                    end

                end
            end

            if minTemp == nil then
                E:trace("Für die Vorhersage konnte keine Tiefsttemperatur ermittelt werden.")
                minTemp = 0
            end
            if maxTemp == nil then
                E:trace("Für die Vorhersage konnte keine Höchsttemperatur ermittelt werden.")
                maxTemp = 0
            end

            -- Allgemeine Tageswerte
            local day = forecastDay["day"]
            day["dMaxTemp_c"]:SetValue(tostring(maxTemp), constValueChange)
            day["dMinTemp_c"]:SetValue(tostring(minTemp), constValueChange)
            day["dRain"]:SetValue(tostring(sumRain), constValueChange)
            day["dSnow"]:SetValue(tostring(sumSnow), constValueChange)
            day["dIcon"]:SetValue(dayIcon, constValueChange)
            day["dHumidity"]:SetValue(dayHumidity, constValueChange)
            day["dPressure"]:SetValue(dayPressure, constValueChange)
            day["dWind_mps"]:SetValue(dayWind, constValueChange)
        end
    end

end

function SortListByDays(decodedJsonTable)
    local days = {}
    local currentDay = nil
    for i=1, getTableLength(decodedJsonTable["list"]), 1 do
        local unixDT = (decodedJsonTable["list"][i]["dt"])
        local dateTime = os.date("%d/%m/%Y %H:%M:%S", unixDT)
        local dateTimeSplit = split(dateTime, " ");

        if currentDay == nil or currentDay.date ~= dateTimeSplit[1] then
            if currentDay ~= nil then
                table.insert(days, currentDay)
            end
            currentDay = { date=dateTimeSplit[1], values={}}
        end
        local values = {}
        values.dateTime = tostring(dateTime)
        values.dMaxTemp_c = tostring(decodedJsonTable["list"][i]["main"]["temp_max"])
        values.dMinTemp_c = tostring(decodedJsonTable["list"][i]["main"]["temp_min"])
        values.dHumidity = tostring(decodedJsonTable["list"][i]["main"]["humidity"])
        values.dPressure = tostring(decodedJsonTable["list"][i]["main"]["pressure"])
        values.dWind_mps =tostring(decodedJsonTable["list"][i]["wind"]["speed"])
        values.dWind_deg = tostring(decodedJsonTable["list"][i]["wind"]["deg"])
        values.dIcon = "https://openweathermap.org/img/wn/" ..tostring(decodedJsonTable["list"][i]["weather"][1]["icon"] .."@2x.png")
        values.dCloud_desc = decodedJsonTable["list"][i]["weather"][1]["description"]
        values.dCloud = decodedJsonTable["list"][i]["clouds"]["all"]
	
		-- Parameter for rain available? 
		if (decodedJsonTable["list"][i]["rain"] == nil) then
			values.dRain = 0
        else
            if (decodedJsonTable["list"][i]["rain"]["3h"] == nil) then
                values.dRain = 0
            else
                values.dRain = tostring(decodedJsonTable["list"][i]["rain"]["3h"])
            end
		end
		
		-- Parameter for snow available? 
		if (decodedJsonTable["list"][i]["snow"] == nil) then
			values.dSnow = 0
        else
            if (decodedJsonTable["list"][i]["snow"]["3h"] == nil) then
                values.dSnow = 0
            else
                values.dSnow = tostring(decodedJsonTable["list"][i]["snow"]["3h"])
            end
        end
        table.insert(currentDay.values, values)
    end

    if currentDay ~= nil then
        table.insert(days,currentDay)
    end
    return days
end


function SetCurrentTable(decodedJsonTable)
    local wCurrentTable = E.PVTable["currentRoot"]
	
	local unixDT = (decodedJsonTable["dt"])
	local DateTime = os.date("%d/%m/%Y %H:%M:%S", unixDT)
	
	wCurrentTable["cLast_updated"]:SetValue(tostring(DateTime), constValueChange)
	wCurrentTable["cTemp_c"]:SetValue(tostring(decodedJsonTable["main"]["temp"]), constValueChange)
	wCurrentTable["cWind_kph"]:SetValue(decodedJsonTable["wind"]["speed"], constValueChange)
    wCurrentTable["cWind_dir"]:SetValue(decodedJsonTable["wind"]["deg"], constValueChange)
    wCurrentTable["cTemp_min"]:SetValue(decodedJsonTable["main"]["temp_min"], constValueChange)
    wCurrentTable["cTemp_max"]:SetValue(decodedJsonTable["main"]["temp_max"], constValueChange)
    wCurrentTable["cPressure_mb"]:SetValue(decodedJsonTable["main"]["pressure"], constValueChange)
    wCurrentTable["cHumidity"]:SetValue(decodedJsonTable["main"]["humidity"], constValueChange)
    wCurrentTable["cCloud"]:SetValue(decodedJsonTable["clouds"]["all"], constValueChange)
	wCurrentTable["cCloud_desc"]:SetValue(decodedJsonTable["weather"][1]["description"], constValueChange)
    wCurrentTable["cIcon"]:SetValue("https://openweathermap.org/img/wn/" ..tostring(decodedJsonTable["weather"][1]["icon"] .."@2x.png"), constValueChange)
    
    return DateTime
end

function SetAstroTable(decodedJsonTable)
	local wAstroTable = E.PVTable["astro"]
	
	local unixDTsunrise = (decodedJsonTable["sys"]["sunrise"])
	local unixDTsunset = (decodedJsonTable["sys"]["sunset"])
	dateTimeSunrise = os.date("%d/%m/%Y %H:%M:%S", unixDTsunrise)
	dateTimeSunset = os.date("%d/%m/%Y %H:%M:%S", unixDTsunset)
	
	AstroBin (dateTimeSunrise, dateTimeSunset)
				
	wAstroTable["aSunrise"]:SetValue(dateTimeSunrise, constValueChange)
    wAstroTable["aSunset"]:SetValue(dateTimeSunset, constValueChange)
	
end
	
function AstroBin (dateTimeSunrise, dateTimeSunset)
	local wAstroTable = E.PVTable["astro"]
	local Systemtime = os.date("%d/%m/%Y %H:%M:%S")
    
    if (nil ~= dateTimeSunrise) and (nil ~= dateTimeSunset) then
        if (Systemtime >= dateTimeSunrise) and (Systemtime < dateTimeSunset)  then
            Astrobinaer = 0
        else
            Astrobinaer = 1
        end
        wAstroTable["aSunUpDown"]:SetValue(Astrobinaer, constValueChange)
    end
end
	
	

-- ===========================================================================
-- DeInitialisierung, einmaliger Aufruf vor dem Beenden des Skripts
-- ===========================================================================

function Exit()
    E:trace("---- HTTP Befehle: Exit Start ----")
   
    -- Schliessen der Ressourcen
    HTTP_RQ:Close()

    E:trace("---- HTTP Befehle: Exit Ende ----")
end

-- ===========================================================================
-- Wird periodisch aufgerufen
-- ===========================================================================

function Poll()

	-- Aktualisiere Anzeige binärer Sonnenauf- / Sonnenuntergang jede Min ----
    if((os.clock() - lastPollTimestampAstro) >= 60) then
        lastPollTimestampAstro = os.clock()
        AstroBin (dateTimeSunrise, dateTimeSunset)
    end
	
    -- Wenn das Intervall fürs automatische aktualisieren == NIL dann Poll abbrechen.
    if (("" == intervallInSeconds) or (nil == intervallInSeconds)) then
        return
    end

    -- Holt die Aktuelle Zeit, zieht den letzten Timestamp ab und wenn das
    -- Ergebnis >= dem "Intervall für das automatische Aktualisieren in Sekunden" entspricht,
    -- dann rufe die Weather API auf und erneuere den Timestamp.
    if((os.clock() - lastPollTimestamp) >= intervallInSeconds) then
        GetValuesOfWeatherAPI()
        lastPollTimestamp = os.clock()
    end
	
end


-- ===========================================================================
-- Ereignisfunktion: Aenderung einer Variablen durch das Prozessmodell
-- ===========================================================================

function OnValueChange( oVarPath, strValue )

    -- E:trace("---- HTTP Befehle: OnValueChange( " .. tostring(oVarPath) .. ", " .. strValue .." ) ----")

    ----------------------------------------------------------------------------
    -- Suche nach dem Datensatz (Tabelle) des sendenden Geraetes
    ----------------------------------------------------------------------------
    local oHTTPDevice = oVarPath:_findParentFromUserType( "httpdevices" )
    if (nil == oHTTPDevice) then
       return
    end
    
    ----------------------------------------------------------------------------
    -- Pruefe, welche Variable sich geaendert hat ...
    ----------------------------------------------------------------------------
    local varChanged = oVarPath:_getLeaf()
    if (nil == varChanged) then
       return
    end

    ----------------------------------------------------------------------------
    -- Pruefe Schreibrechte
    ----------------------------------------------------------------------------

    local nAccessRights = varChanged:GetAccessRights()
    if ((nAccessRights ~= constAccessReadWrite) and (nAccessRights ~= constAccessWrite)) then
        return
    end
	
	----------------------------------------------------------------------------
    -- Pruefe ob der Benutzer die WeatherAPI manuell aktualisieren moechte
    ----------------------------------------------------------------------------
	
	if (varChanged == oHTTPDevice["manuelUpdateWeather"]) then
		GetValuesOfWeatherAPI()
		return
    end

    ----------------------------------------------------------------------------
    -- Pruefe ob der Benutzer das Intervall für das automatische Aktualisieren 
	-- geaendert hat
    ----------------------------------------------------------------------------
    
    if (varChanged == oHTTPDevice["pollSeconds"]) then
        if (cvtStringToNumber(strValue) < 60) then
            strValue = 60
        end
        intervallInSeconds = cvtStringToNumber(strValue)
        oHTTPDevice["pollSeconds"]:SetValue(intervallInSeconds, constValueChange)
        lastPollTimestamp = os.clock()
	end	
	
end

-- ===========================================================================
-- Ereignisfunktion: Abfrage eines Variablenwertes durch das Prozessmodell
--                   Gebe aktuellen Wert zurück, falls vorhanden
-- ===========================================================================

function OnValueRead( oVarPath, nReason )

    ----------------------------------------------------------------------------
    -- Pr�fe, welcher Datenpunkt gelesen werden soll
    ----------------------------------------------------------------------------

    local varToRead = oVarPath:_getLeaf()
    if (nil == varToRead) then
        return
    end

    -- Prüfe Leserechte für Datenpunkt
    local nAccessRights = varToRead:GetAccessRights()
    if ((nAccessRights ~= constAccessRead) and (nAccessRights ~= constAccessReadWrite)) then
        return
    end

    -- Sende Wert an Prozessmodell, sofern vorhanden
    local strAvailableValue = varToRead:GetValue()
    if (nil ~= strAvailableValue) then
        varToRead:SetValue( strAvailableValue, constResponseFromCache )
    end
end


